<?php
// =================================================================
// KONFIGURACJA I LOGIKA APLIKACJI
// =================================================================

// --- USTAWIENIA BEZPIECZEŃSTWA ---
// Definiujemy główny katalog, od którego skrypt ma działać.
// __DIR__ to stała PHP zawierająca ścieżkę do katalogu, w którym znajduje się ten skrypt.
// Dzięki temu ograniczamy dostęp tylko do plików w tym folderze i jego podfolderach.
 $baseDir = __DIR__;

// --- OBSŁUGA PARAMETRÓW Z URL ---

// Pobranie aktualnego katalogu z parametru 'dir' w URL.
// Jeśli brak parametru, domyślnie ustawiamy katalog bieżący ('.').
// W tej wersji 'dir' nie jest używany do nawigacji, ale może być przydatny w przyszłości.
 $currentDir = isset($_GET['dir']) ? $_GET['dir'] : '.';

// Pobranie ścieżki do pliku z parametrem 'file' w URL.
 $requestedFile = isset($_GET['file']) ? $_GET['file'] : null;

// --- PRZETWARZANIE I GENEROWANIE HTML ---

// Zmienna przechowująca HTML dla lewego panelu (przeglądarki plików)
 $fileBrowserHtml = '';

// Zmienna przechowująca HTML dla prawego panelu (podglądu zawartości)
 $contentViewerHtml = '';

// Zmienna przechowująca HTML dla przycisków nawigacyjnych
 $navigationHtml = '';

// Budujemy pełną, bezpieczną ścieżkę do systemu plików
 $fullPath = realpath($baseDir . '/' . $currentDir);

// --- KONTROLA BEZPIECZEŃSTWA ---
// Sprawdzamy, czy ścieżka jest prawidłowa i czy znajduje się wewnątrz naszego katalogu bazowego.
// To kluczowe zabezpieczenie przed dostępem do innych części serwera.
if ($fullPath === false || strpos($fullPath, $baseDir) !== 0) {
    // Jeśli ścieżka jest nieprawidłowa lub próbuje wyjść poza bazę, przerywamy działanie.
    die('<h1 style="color:red;">Błąd: Odmowa dostępu.</h1>');
}

// Funkcja do rekurencyjnego skanowania katalogów i tworzenia drzewa plików
function buildFileTree($dir, $baseDir, $currentFile = null) {
    $html = '';
    $items = scandir($dir);
    
    // Sortujemy elementy - najpierw katalogi, potem pliki
    $dirs = [];
    $files = [];
    
    foreach ($items as $item) {
        if ($item == '.' || $item == '..') {
            continue;
        }
        
        $itemPath = $dir . '/' . $item;
        
        if (is_dir($itemPath)) {
            // Sprawdzamy, czy folder nie zaczyna się od kropki (niewidzialny folder)
            if (substr($item, 0, 1) !== '.') {
                $dirs[] = $item;
            }
        } elseif (is_file($itemPath)) {
            $extension = strtolower(pathinfo($item, PATHINFO_EXTENSION));
            if (in_array($extension, ['html', 'htm', 'php', 'pdf', 'txt', 'mp4'])) {
                // Wykluczamy główny plik index.php z listy, aby uniknąć rekurencji
                if ($item === 'index.php') {
                    continue;
                }
                $files[] = $item;
            }
        }
    }
    
    sort($dirs);
    sort($files);
    
    foreach ($dirs as $item) {
        $itemPath = $dir . '/' . $item;
        $relativePath = str_replace($baseDir . '/', '', $itemPath);
        
        $html .= '<div class="folder-item">';
        $html .= '<div class="folder-header" onclick="toggleFolder(this)">';
        $html .= '<span class="folder-toggle">▼</span>';
        $html .= '<span class="folder-name">' . htmlspecialchars($item) . '</span>';
        $html .= '</div>';
        $html .= '<div class="folder-content">';
        $html .= buildFileTree($itemPath, $baseDir, $currentFile);
        $html .= '</div>';
        $html .= '</div>';
    }
    
    foreach ($files as $item) {
        $itemPath = $dir . '/' . $item;
        $relativePath = str_replace($baseDir . '/', '', $itemPath);
        $fileNameWithoutExt = pathinfo($item, PATHINFO_FILENAME);
        
        $activeClass = ($currentFile && $relativePath === $currentFile) ? ' active' : '';
        
        $html .= '<div class="file-item' . $activeClass . '">';
        $html .= '<a href="?file=' . urlencode($relativePath) . '" class="file-link">' . htmlspecialchars($fileNameWithoutExt) . '</a>';
        $html .= '</div>';
    }
    
    return $html;
}

// Funkcja do znalezienia poprzedniego i następnego pliku
function findAdjacentFiles($currentFile, $baseDir) {
    if (!$currentFile) {
        return ['prev' => null, 'next' => null];
    }
    
    $currentFilePath = realpath($baseDir . '/' . $currentFile);
    if ($currentFilePath === false) return ['prev' => null, 'next' => null];
    
    $currentDir = dirname($currentFilePath);
    
    $allFiles = [];
    $items = scandir($currentDir);
    
    foreach ($items as $item) {
        if ($item == '.' || $item == '..') {
            continue;
        }
        
        $itemPath = $currentDir . '/' . $item;
        
        if (is_file($itemPath)) {
            $extension = strtolower(pathinfo($item, PATHINFO_EXTENSION));
            if (in_array($extension, ['html', 'htm', 'php', 'pdf', 'txt', 'mp4'])) {
                // Wykluczamy główny plik index.php również z nawigacji
                if ($item === 'index.php') {
                    continue;
                }
                $relativePath = str_replace($baseDir . '/', '', $itemPath);
                $allFiles[] = $relativePath;
            }
        }
    }
    
    sort($allFiles);
    
    $currentIndex = array_search($currentFile, $allFiles);
    
    $prev = ($currentIndex > 0) ? $allFiles[$currentIndex - 1] : null;
    $next = ($currentIndex < count($allFiles) - 1) ? $allFiles[$currentIndex + 1] : null;
    
    return ['prev' => $prev, 'next' => $next];
}

// --- LOGIKA PRZEGLĄDARKI PLIKÓW (LEWY PANEL) ---
// Budowanie drzewa plików
 $fileTreeHtml = buildFileTree($baseDir, $baseDir, $requestedFile);

// --- LOGIKA PRZYCISKÓW NAWIGACYJNYCH ---
 $adjacentFiles = findAdjacentFiles($requestedFile, $baseDir);

 $navigationHtml .= '<div class="navigation-buttons">';
 $navigationHtml .= '<a href="#" class="nav-button"';
if ($adjacentFiles['prev']) {
    $navigationHtml .= ' onclick="window.location.href=\'?file=' . urlencode($adjacentFiles['prev']) . '\'"';
} else {
    $navigationHtml .= ' disabled';
}
 $navigationHtml .= '>&lt;WSTECZ</a>';

 $navigationHtml .= '<a href="index.php" class="nav-button">GŁÓWNA</a>';

 $navigationHtml .= '<a href="#" class="nav-button"';
if ($adjacentFiles['next']) {
    $navigationHtml .= ' onclick="window.location.href=\'?file=' . urlencode($adjacentFiles['next']) . '\'"';
} else {
    $navigationHtml .= ' disabled';
}
 $navigationHtml .= '>NASTĘPNA&gt;</a>';
 $navigationHtml .= '</div>';

// --- LOGIKA PODGLĄDU ZAWARTOŚCI (PRAWY PANEL) ---
if ($requestedFile) {
    // Budujemy pełną ścieżkę do żądanego pliku
    $fullFilePath = realpath($baseDir . '/' . $requestedFile);
    $extension = strtolower(pathinfo($fullFilePath, PATHINFO_EXTENSION));

    // Dodajemy warunek, aby nigdy nie próbować dołączać samego siebie
    $scriptName = basename($_SERVER['SCRIPT_NAME']);
    if ($fullFilePath && strpos($fullFilePath, $baseDir) === 0 && is_file($fullFilePath) && basename($fullFilePath) !== $scriptName) {
        
        if ($extension === 'pdf') {
            // *** LOGIKA WARUNKOWA: WYBÓR METODY WYŚWIETLANIA PDF ***
            $isMobile = (bool)preg_match('/(android|avantgo|blackberry|bolt|boost|cricket|docomo|fone|hiptop|mini|mobi|palm|phone|pie|tablet|up\.browser|up\.link|webos|wos)/i', $_SERVER['HTTP_USER_AGENT']);

            $contentViewerHtml .= '<div class="file-content pdf-container">';
            
            if ($isMobile) {
                // --- WERSJA MOBILNA: Używamy PDF.js ---
                $contentViewerHtml .= '<div class="pdf-controls mobile-pdf-controls">';
                $contentViewerHtml .= '<div class="zoom-controls">';
                $contentViewerHtml .= '<button id="zoom-out" class="zoom-button">-</button>';
                $contentViewerHtml .= '<input type="range" id="zoom-slider" class="zoom-slider" min="50" max="200" value="100">';
                $contentViewerHtml .= '<button id="zoom-in" class="zoom-button">+</button>';
                $contentViewerHtml .= '<span id="zoom-level" class="zoom-level">100%</span>';
                $contentViewerHtml .= '</div>';
                $contentViewerHtml .= '<div class="page-controls">';
                $contentViewerHtml .= '<button id="prev-page" class="page-button">&lt;</button>';
                $contentViewerHtml .= '<span id="page-info" class="page-info">0 / 0</span>';
                $contentViewerHtml .= '<button id="next-page" class="page-button">&gt;</button>';
                $contentViewerHtml .= '</div>';
                $contentViewerHtml .= '</div>';
                $contentViewerHtml .= '<div id="pdf-canvas-container" class="pdf-canvas-container">';
                $contentViewerHtml .= '<canvas id="pdf-canvas"></canvas>';
                $contentViewerHtml .= '<div id="pdf-loading" class="pdf-loading">Ładowanie dokumentu...</div>';
                $contentViewerHtml .= '</div>';
            } else {
                // --- WERSJA DESKTOPOWA: Używamy natywnego iframe BEZ kontrolek ---
                $contentViewerHtml .= '<div class="pdf-iframe-container">';
                $contentViewerHtml .= '<iframe src="' . htmlspecialchars($requestedFile) . '" class="pdf-iframe" id="pdf-viewer" style="border: none;"></iframe>';
                $contentViewerHtml .= '</div>';
            }

            $contentViewerHtml .= '</div>';
        } elseif ($extension === 'php') {
            $contentViewerHtml .= '<div class="file-content php-content">';
            $iframeId = 'php-iframe-' . uniqid();
            $contentViewerHtml .= '<iframe id="' . $iframeId . '" src="' . htmlspecialchars($requestedFile) . '" class="php-iframe" style="border: none;" onload="applyStylesToIframe(\'' . $iframeId . '\')"></iframe>';
            $contentViewerHtml .= '</div>';
        } elseif ($extension === 'txt') {
            $txtContent = file_get_contents($fullFilePath);
            $contentViewerHtml .= '<div class="file-content">' . nl2br(htmlspecialchars($txtContent)) . '</div>';
        } elseif ($extension === 'mp4') {
            $contentViewerHtml .= '<div class="file-content">';
            $contentViewerHtml .= '<video width="calc(100% - 4px)" height="auto" controls style="margin: 2px;">';
            $contentViewerHtml .= '<source src="' . htmlspecialchars($requestedFile) . '" type="video/mp4">';
            $contentViewerHtml .= 'Twoja przeglądarka nie obsługuje odtwarzacza wideo.';
            $contentViewerHtml .= '</video>';
            $contentViewerHtml .= '</div>';
        } else {
            $htmlContent = file_get_contents($fullFilePath);
            $htmlContent = preg_replace('/<!DOCTYPE[^>]*>/', '', $htmlContent);
            $htmlContent = preg_replace('/<html[^>]*>/', '', $htmlContent);
            $htmlContent = preg_replace('/<\/html>/', '', $htmlContent);
            $htmlContent = preg_replace('/<head[^>]*>.*?<\/head>/is', '', $htmlContent);
            $htmlContent = preg_replace('/<body[^>]*>/', '', $htmlContent);
            $htmlContent = preg_replace('/<\/body>/', '', $htmlContent);
            $htmlContent = preg_replace('/<style[^>]*>.*?<\/style>/is', '', $htmlContent);
            $htmlContent = preg_replace('/<link[^>]*>/is', '', $htmlContent);
            $htmlContent = preg_replace('/<script[^>]*>.*?<\/script>/is', '', $htmlContent);
            
            if (function_exists('iconv')) {
                if (preg_match('/charset=([^\s">]+)/i', $htmlContent, $matches)) {
                    $detectedEncoding = strtoupper($matches[1]);
                    if ($detectedEncoding !== 'UTF-8') {
                        $htmlContent = iconv($detectedEncoding, 'UTF-8//IGNORE', $htmlContent);
                        $htmlContent = preg_replace('/charset=[^\s">]+/i', 'charset=UTF-8', $htmlContent);
                    }
                } else {
                    $encodingsToTry = ['ISO-8859-2', 'WINDOWS-1250', 'UTF-8'];
                    foreach ($encodingsToTry as $encoding) {
                        $converted = @iconv($encoding, 'UTF-8//IGNORE', $htmlContent);
                        if ($converted !== false && strlen($converted) > 0) {
                            $htmlContent = $converted;
                            break;
                        }
                    }
                }
            }
            
            $contentViewerHtml .= '<div class="file-content">' . $htmlContent . '</div>';
        }
    } else {
        $contentViewerHtml = '<p style="color: #ff4d4d; font-weight: bold;">Błąd: Nie można załadować wskazanego pliku lub plik nie istnieje.</p>';
    }
} else {
    $contentViewerHtml = '<div class="welcome-message">';
    $contentViewerHtml .= '<h2>Dzień Dobry :)</h2>';
    $contentViewerHtml .= '<p>Wybierz proszę stronę z lewego panelu, aby rozpocząć przeglądanie.</p>';
    $contentViewerHtml .= '</div>';
}
?>
<!DOCTYPE html>
<html lang="pl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>A'la Blog - Michał T.</title>
    <!-- Biblioteka PDF.js jest ładowana, ale używana tylko na mobile -->
    <script src="https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.min.js"></script>
    <style>
        /* --- PODSTAWOWE STYLIZOWANIE --- */
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: "Courier New", Courier, monospace;
            margin: 0;
            padding: 0;
            background-color: #000000;
            color: #00ff00;
            font-size: 16px;
        }

        header {
            background-color: #1a1a1a;
            color: #00ff00;
            padding: 1rem;
            text-align: center;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            border-bottom: 1px solid #00ff00;
            position: relative;
        }

        header h1 {
            margin: 0;
            font-size: 1.8rem;
        }
        
        .menu-toggle {
            display: none;
            position: absolute;
            left: 15px;
            top: 50%;
            transform: translateY(-50%);
            background: none;
            border: 1px solid #00ff00;
            color: #00ff00;
            font-size: 1.5rem;
            padding: 5px 10px;
            cursor: pointer;
            border-radius: 4px;
        }

        main {
            display: flex;
            height: calc(100vh - 80px);
        }

        /* --- LEWY PANEL --- */
        #file-browser {
            width: 300px;
            flex-shrink: 0;
            background-color: #111111;
            border-right: 1px solid #00ff00;
            padding: 0;
            overflow-y: auto;
            box-sizing: border-box;
            transition: transform 0.3s ease;
            display: flex;
            flex-direction: column;
        }

        .navigation-buttons {
            display: flex;
            justify-content: space-between;
            padding: 10px;
            border-bottom: 1px solid #00ff00;
            background-color: #1a1a1a;
            flex-shrink: 0;
        }

        .nav-button {
            display: inline-block;
            padding: 8px 12px;
            background-color: #333;
            color: #00ff00;
            border: 1px solid #00ff00;
            text-decoration: none;
            border-radius: 4px;
            transition: background-color 0.2s;
            font-weight: bold;
            font-size: 0.9rem;
            text-align: center;
            flex-grow: 1;
            margin: 0 2px;
        }

        .nav-button:hover:not([disabled]) {
            background-color: #00ff00;
            color: #000;
        }

        .nav-button[disabled] {
            color: #666;
            border-color: #666;
            cursor: not-allowed;
            background-color: transparent;
        }

        .file-tree {
            padding: 1rem;
            overflow-y: auto;
            flex-grow: 1;
        }

        .folder-item {
            margin-bottom: 5px;
        }

        .folder-header {
            display: flex;
            align-items: center;
            padding: 5px;
            cursor: pointer;
            border-radius: 4px;
            transition: background-color 0.2s;
        }

        .folder-header:hover {
            background-color: #333333;
        }

        .folder-toggle {
            margin-right: 5px;
            width: 15px;
            text-align: center;
        }

        .folder-content {
            margin-left: 20px;
            display: block;
        }

        .folder-content.collapsed {
            display: none;
        }

        .file-item {
            margin-bottom: 2px;
        }

        .file-item.active {
            background-color: #333;
        }

        .file-link {
            display: block;
            padding: 5px;
            text-decoration: none;
            color: #00ff00;
            border-radius: 4px;
            transition: background-color 0.2s;
        }

        .file-link:hover {
            background-color: #333333;
        }

        /* --- PRAWY PANEL --- */
        #content-viewer {
            flex-grow: 1;
            padding: 1.5rem;
            overflow-y: auto;
            background-color: #000000;
        }

        .file-content {
            width: 100%;
            height: auto;
            overflow: hidden;
            min-height: 90vh;
            display: flex;
            flex-direction: column;
        }

        .file-content * {
            max-width: 100% !important;
            box-sizing: border-box !important;
        }

        .php-iframe {
            width: 100% !important;
            height: 90vh !important;
            border: none !important;
            flex-grow: 1;
        }

        /* --- STYLE DLA PDF (WSPÓLNE I DEDYKOWANE) --- */
        .pdf-container {
            display: flex;
            flex-direction: column;
            height: 100%;
        }

        /* Kontrolek mobilne są domyślnie ukryte */
        .mobile-pdf-controls {
            display: none;
        }
        
        .zoom-controls, .page-controls {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .page-button {
            background-color: #333;
            color: #00ff00;
            border: 1px solid #00ff00;
            width: 30px;
            height: 30px;
            border-radius: 4px;
            font-size: 16px;
            font-weight: bold;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .page-button:hover:not([disabled]) {
            background-color: #00ff00;
            color: #000;
        }

        .page-button[disabled] {
            color: #666;
            border-color: #666;
            cursor: not-allowed;
        }

        .page-info {
            color: #00ff00;
            font-weight: bold;
            min-width: 60px;
            text-align: center;
        }

        .zoom-button {
            background-color: #333;
            color: #00ff00;
            border: 1px solid #00ff00;
            width: 30px;
            height: 30px;
            border-radius: 4px;
            font-size: 16px;
            font-weight: bold;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .zoom-button:hover {
            background-color: #00ff00;
            color: #000;
        }

        .zoom-slider {
            width: 150px;
            height: 5px;
            background: #333;
            outline: none;
            border-radius: 5px;
        }

        .zoom-slider::-webkit-slider-thumb {
            -webkit-appearance: none;
            appearance: none;
            width: 20px;
            height: 20px;
            background: #00ff00;
            cursor: pointer;
            border-radius: 50%;
        }

        .zoom-slider::-moz-range-thumb {
            width: 20px;
            height: 20px;
            background: #00ff00;
            cursor: pointer;
            border-radius: 50%;
        }

        .zoom-level {
            color: #00ff00;
            font-weight: bold;
            min-width: 45px;
            text-align: center;
        }

        /* Style dla widoku desktopowego (iframe) */
        .pdf-iframe-container {
            flex-grow: 1;
            overflow: auto;
            position: relative;
            background-color: #000;
        }

        #pdf-viewer {
            width: 100%;
            height: 100%;
            border: none;
            display: block;
        }

        /* Style dla widoku mobilnego (PDF.js) */
        .pdf-canvas-container {
            flex-grow: 1;
            overflow: auto;
            display: flex;
            justify-content: center;
            align-items: flex-start;
            background-color: #333;
            position: relative;
        }

        #pdf-canvas {
            border: 1px solid #00ff00;
            box-shadow: 0 0 10px rgba(0, 255, 0, 0.3);
            display: block;
            max-width: none !important; /* Kluczowe dla poprawnego zoomu */
        }
        
        .pdf-loading {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            font-size: 1.2rem;
            font-weight: bold;
        }

        /* --- RESPONSYWNOŚĆ --- */
        @media screen and (max-width: 768px) {
            /* Na mobile pokazujemy kontrolki mobilne */
            .mobile-pdf-controls {
                display: flex;
                flex-direction: column;
                align-items: stretch;
                gap: 10px;
                padding: 10px;
                background-color: #1a1a1a;
                border-bottom: 1px solid #00ff00;
            }
            .mobile-pdf-controls .zoom-controls, .mobile-pdf-controls .page-controls {
                justify-content: center;
            }
            .pdf-canvas-container {
                height: calc(100% - 120px); /* Adjust for taller controls */
            }

            .menu-toggle {
                display: block;
            }
            
            #file-browser {
                position: absolute;
                left: 0;
                top: 80px;
                height: calc(100vh - 80px);
                z-index: 1000;
                transform: translateX(-100%);
            }
            
            #file-browser.active {
                transform: translateX(0);
            }
            
            #content-viewer {
                width: 100%;
                padding: 1rem;
            }
            
            main {
                flex-direction: column;
            }
            
            header h1 {
                font-size: 1.4rem;
                padding-left: 40px;
            }
            
            .nav-button {
                padding: 6px 8px;
                font-size: 0.8rem;
            }
            
            .php-iframe {
                width: 100% !important;
                height: 80vh !important;
                min-height: 400px !important;
            }
            
            .file-content {
                min-height: 80vh;
            }
            
            .zoom-slider {
                width: 120px;
            }
        }
        
        @media screen and (max-width: 480px) {
            body {
                font-size: 14px;
            }
            
            header h1 {
                font-size: 1.2rem;
            }
            
            #file-browser {
                width: 100%;
            }
            
            #content-viewer {
                padding: 0.8rem;
            }
            
            .nav-button {
                padding: 5px 6px;
                font-size: 0.7rem;
            }
            
            .php-iframe {
                width: 100% !important;
                height: 70vh !important;
                min-height: 300px !important;
            }
            
            .file-content {
                min-height: 70vh;
            }
            
            .zoom-controls {
                gap: 5px;
            }
            
            .zoom-slider {
                width: 100px;
            }
            
            .zoom-button {
                width: 25px;
                height: 25px;
                font-size: 14px;
            }
        }
        
        /* --- STYLE DLA HTML/PHP IFRAME --- */
        .file-content input[type="text"],
        .file-content input[type="password"],
        .file-content input[type="email"],
        .file-content input[type="search"],
        .file-content input[type="number"],
        .file-content input[type="tel"],
        .file-content input[type="url"],
        .file-content input[type="date"],
        .file-content input[type="datetime-local"],
        .file-content input[type="month"],
        .file-content input[type="time"],
        .file-content input[type="week"],
        .file-content textarea,
        .file-content select {
            background-color: white !important;
            color: black !important;
            border: 1px solid #ccc !important;
        }
        
        .file-content button,
        .file-content input[type="button"],
        .file-content input[type="submit"],
        .file-content input[type="reset"] {
            background-color: #d3d3d3 !important;
            color: black !important;
            border: 1px solid #999 !important;
        }
        
        .file-content a,
        .file-content a:visited {
            color: #00ff00 !important;
            text-shadow: 0 0 3px rgba(255, 255, 255, 0.3) !important;
        }
    </style>
</head>
<body>

    <header>
        <button class="menu-toggle" id="menuToggle">☰</button>
        <h1>A'la Blog: Wpisy, poradniki i polecajki.</h1>
    </header>

    <main>
        <nav id="file-browser">
            <?php echo $navigationHtml; ?>
            <div class="file-tree">
                <?php echo $fileTreeHtml; ?>
            </div>
        </nav>

        <section id="content-viewer">
            <?php echo $contentViewerHtml; ?>
        </section>
    </main>

    <script>
        // Konfiguracja PDF.js (używana tylko na mobile)
        pdfjsLib.GlobalWorkerOptions.workerSrc = 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.worker.min.js';

        // Zmienne globalne dla PDF.js
        let pdfDoc = null;
        let pageNum = 1;
        let pageRendering = false;
        let pageNumPending = null;
        let scale = 1.0;

        document.addEventListener('DOMContentLoaded', function() {
            const menuToggle = document.getElementById('menuToggle');
            const fileBrowser = document.getElementById('file-browser');
            
            menuToggle.addEventListener('click', function() {
                fileBrowser.classList.toggle('active');
            });
            
            const links = document.querySelectorAll('.file-link');
            links.forEach(link => {
                link.addEventListener('click', function() {
                    if (window.innerWidth <= 768) {
                        fileBrowser.classList.remove('active');
                    }
                });
            });
            
            // Sprawdź, który widok PDF zainicjalizować
            if (document.getElementById('pdf-canvas')) {
                // Widok mobilny (PDF.js)
                initPdfJsViewer();
            }
            // Funkcja initNativePdfZoom() została usunięta, bo nie jest już potrzebna
        });

        function toggleFolder(element) {
            const content = element.nextElementSibling;
            const toggle = element.querySelector('.folder-toggle');
            
            if (content.classList.contains('collapsed')) {
                content.classList.remove('collapsed');
                toggle.textContent = '▼';
            } else {
                content.classList.add('collapsed');
                toggle.textContent = '▶';
            }
        }
        
        // --- FUNKCJE DLA WIDOKU MOBILNEGO (PDF.JS) ---
        function initPdfJsViewer() {
            const url = '<?php echo htmlspecialchars($requestedFile); ?>';
            const canvas = document.getElementById('pdf-canvas');
            const ctx = canvas.getContext('2d');
            const loadingIndicator = document.getElementById('pdf-loading');

            const savedZoom = localStorage.getItem('pdfZoom');
            scale = savedZoom ? parseFloat(savedZoom) / 100 : 1.0;
            updateZoomControls(scale);

            function renderPage(num) {
                pageRendering = true;
                pdfDoc.getPage(num).then(function(page) {
                    const viewport = page.getViewport({scale: scale});
                    canvas.height = viewport.height;
                    canvas.width = viewport.width;
                    const renderContext = { canvasContext: ctx, viewport: viewport };
                    const renderTask = page.render(renderContext);
                    renderTask.promise.then(function() {
                        pageRendering = false;
                        if (pageNumPending !== null) {
                            renderPage(pageNumPending);
                            pageNumPending = null;
                        }
                    });
                });
                document.getElementById('page-info').textContent = `${num} / ${pdfDoc.numPages}`;
                updatePageButtons(num);
            }

            function queueRenderPage(num) {
                if (pageRendering) {
                    pageNumPending = num;
                } else {
                    renderPage(num);
                }
            }

            function onPrevPage() {
                if (pageNum <= 1) return;
                pageNum--;
                queueRenderPage(pageNum);
            }
            document.getElementById('prev-page').addEventListener('click', onPrevPage);

            function onNextPage() {
                if (pageNum >= pdfDoc.numPages) return;
                pageNum++;
                queueRenderPage(pageNum);
            }
            document.getElementById('next-page').addEventListener('click', onNextPage);

            function updateZoom(newScale) {
                scale = newScale;
                localStorage.setItem('pdfZoom', Math.round(scale * 100));
                updateZoomControls(scale);
                if (pdfDoc) {
                    queueRenderPage(pageNum);
                }
            }

            function updateZoomControls(currentScale) {
                const zoomPercent = Math.round(currentScale * 100);
                document.getElementById('zoom-slider').value = zoomPercent;
                document.getElementById('zoom-level').textContent = zoomPercent + '%';
            }

            document.getElementById('zoom-slider').addEventListener('input', function() {
                updateZoom(parseInt(this.value) / 100);
            });

            document.getElementById('zoom-in').addEventListener('click', function() {
                updateZoom(Math.min(scale + 0.1, 2.0));
            });

            document.getElementById('zoom-out').addEventListener('click', function() {
                updateZoom(Math.max(scale - 0.1, 0.5));
            });

            function updatePageButtons(currentPageNum) {
                document.getElementById('prev-page').disabled = currentPageNum <= 1;
                document.getElementById('next-page').disabled = currentPageNum >= pdfDoc.numPages;
            }

            pdfjsLib.getDocument(url).promise.then(function(pdfDoc_) {
                pdfDoc = pdfDoc_;
                document.getElementById('page-info').textContent = `1 / ${pdfDoc.numPages}`;
                updatePageButtons(pageNum);
                loadingIndicator.style.display = 'none';
                renderPage(pageNum);
            }).catch(function(error) {
                console.error('Błąd podczas ładowania PDF:', error);
                loadingIndicator.textContent = 'Błąd ładowania dokumentu.';
                loadingIndicator.style.color = '#ff4d4d';
            });
        }
        
        // Funkcja do stylowania PHP iframe
        function applyStylesToIframe(iframeId) {
            try {
                const iframe = document.getElementById(iframeId);
                if (!iframe) return;
                
                iframe.onload = function() {
                    try {
                        const iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
                        const style = iframeDoc.createElement('style');
                        style.textContent = `
                            body { background-color: #000000 !important; color: #00ff00 !important; font-family: "Courier New", Courier, monospace !important; }
                            * { color: #00ff00 !important; }
                            input[type="text"], input[type="password"], input[type="email"], input[type="search"], 
                            input[type="number"], input[type="tel"], input[type="url"], input[type="date"], 
                            input[type="datetime-local"], input[type="month"], input[type="time"], input[type="week"], 
                            textarea, select {
                                background-color: white !important;
                                color: black !important;
                                border: 1px solid #ccc !important;
                            }
                            button, input[type="button"], input[type="submit"], input[type="reset"] {
                                background-color: #d3d3d3 !important;
                                color: black !important;
                                border: 1px solid #999 !important;
                            }
                            a, a:visited {
                                color: #00ff00 !important;
                                text-shadow: 0 0 3px rgba(255, 255, 255, 0.3) !important;
                            }
                        `;
                        if (iframeDoc.head) {
                            iframeDoc.head.appendChild(style);
                        } else if (iframeDoc.documentElement) {
                            const head = iframeDoc.createElement('head');
                            iframeDoc.documentElement.insertBefore(head, iframeDoc.documentElement.firstChild);
                            head.appendChild(style);
                        }
                        if (!iframeDoc.body && iframeDoc.documentElement) {
                            const body = iframeDoc.createElement('body');
                            iframeDoc.documentElement.appendChild(body);
                        }
                    } catch (e) {
                        console.error('Błąd podczas modyfikacji stylów w iframe:', e);
                    }
                };
                
                if (iframe.contentDocument && iframe.contentDocument.readyState === 'complete') {
                    iframe.onload();
                }
            } catch (e) {
                console.error('Błąd podczas dostępu do iframe:', e);
            }
        }
    </script>
</body>
</html>